﻿; HTML Playground - Editeur HTML/JavaScript
; Version 2.01

; PureBasic 6.30 (x64)
; Mise à jour du 5 Février 2026
; Pas d'API & Scintilla natif

; Fonctionnalités
; ===============
; -Un éditeur de code HTML/CSS/Javascript
; -Voir le résultat dans un WebViewGadget

; Todo List
; Sauvegardes du code
; Utilitaires choix des couleurs
; Ajout de mots clés
; Améliorer l'auto complétion (Afficher uniquement les mots qui commence par les premieres lettres de la saisie)
;

EnableExplicit

XIncludeFile "keywords.pbi"
XIncludeFile "TabBarGadget.pbi"

;{ Enumération
Enumeration window
  #app
  #StatusBar
EndEnumeration

Enumeration gadgets 
  #TB
  
  #editor
  #webView 
EndEnumeration

Enumeration Menu
  #MainMenu
  
  #FileNew
  #FileOpen
  #FileSave
  #FileSaveAs
  
  #EditCut
  #EditCopy
  #EditPaste
  
  #Quit
  
  #MenuPop
  #MenuPopCut
  #MenuPopCopy
  #MenuPopPaste
EndEnumeration

Enumeration styles
  #Style_HTML_Comment
  #Style_HTML_Keyword
  
  #Style_JS_Comment
  #Style_JS_KeyWord
  #Style_JS_Number
  #Style_JS_String 
  #Style_JS_Bracket
  #Style_JS_Parentheses
  
  #Style_CSS_Selector
  #Style_CSS_Property
  #Style_CSS_Value
  #Style_CSS_Number
  #Style_CSS_String
  #Style_CSS_Comment
  #Style_CSS_Punct
EndEnumeration
;}

;{ Variables 
Define version.s = "2.01"

Define windowStyle.i=#PB_Window_MinimizeGadget|#PB_Window_MaximizeGadget|#PB_Window_ScreenCentered|#PB_Window_SizeGadget

Global tabCurrentId 
Global tabPreviousId

; Largeur de la marge de numérotation des lignes
Global scMarginSizeDefault = 50

; Police de caractére
Global scFont.s = "Lucida Console"
Global scFontSize = 12

; Couleurs pardéfaut de l'éditeur Scintilla
Global scFrontColor = RGB(0, 0, 0) 
Global scBackColor  = RGB(217, 217, 217)

Global scCurrentLineColor = RGB(255, 228, 181)
Global scMasterFolder = RGB(255, 0, 0)

; Couleur des styles par defaut
Global scStyleHTMLKeyWordColor = RGB(255, 0, 0)
Global scStyleHTMLCommentColor = RGB(0, 255, 0)

Global scStyleJSCommentColor = RGB(138, 186, 90)
Global scStyleJSKeywordColor = RGB(0, 0, 205)
Global scStyleJSNumber = RGB(0, 180, 200)
Global scStyleJSString = RGB(136, 201, 144)
Global scStyleJSSBracket = RGB(0, 0, 255)
Global scStyleJSParentheses = RGB(255, 0, 0)

Global scStyleCSSKeywordColor = RGB(0, 0, 205)
Global scStyleCSSProperty = RGB(0, 120, 255)
Global scStyleCSSValue = RGB(0, 170, 120)
Global scStyleCSSNumber = RGB(0, 200, 200)
Global scStyleCSSString = RGB(180, 0, 180)
Global scStyleCSSComment = RGB(0, 150, 0)
Global scStyleCSSPunct = RGB(200, 200, 0)

; Indentation
Global scIndent, scLine, scPos, scCol

; Pour la recherche des blocs commentaires "<!--" et "-->"
; Ces deux chaines doivent être stockées dans un buffer UTF8
Global commentStartPattern = AllocateMemory(5)
Global commentEndPattern   = AllocateMemory(4)

Global commentCSSStartPattern = AllocateMemory(5)
Global commentCSSEndPattern   = AllocateMemory(4)

Structure newProject
  name.s
  html.s
EndStructure
Global currentProject.newProject

; Variables de la fenetre
Global MarginBot = 96 
Global font = LoadFont(#PB_Any, "Arial", 10)
Global windowColor = RGB(206, 206, 206) 
;}

;- Sommaire des procédures
;{ Sommaire des procédures
; Initialisation d'un nouveau orojet
Declare newProject()

; Définir les paramétres du gadget scintilla
Declare scintillaProperties(gadget)

; Callback évenementielle du gadget scintilla 
Declare ScintillaCallBack(gadget, *scn.scNotification)

; Obtenir le texte de la ligne en cours de saisie 
Declare.s GetScintillaLineText(gadget, line)

; Obtenir le mot se trouvant entre deux position
Declare.s getScintillaWord(gadget, startPos, endPos)

; Tester si un mot est un mot clé
Declare.b isKeyword(word.s, keyWords.s)

; Déterminer le string précédent le curseur de saisie
Declare.s getOpeningTagBeforeCaret(gadget)

; Tester si un mot est une clé de pliage
Declare.b isFoldingKeyWord(word.s, keyWordfoldingDown.s)

; Retourne les positions de début et fin du code JavaScript
Declare getJSRange(gadget, *start.Integer, *end.Integer, currentPos)

; Coloration des mots clés HTML
Declare colorizeHTMLTags(gadget, startPos, endPos, keyWords.s)

; Ignorer les lignes de commentaires JavaScript "//"
Declare skipJSLineComment(gadget, pos)

; Ignorer les blocs de commentaires JavaScript "/* .. */"
Declare skipJSBlockComment(gadget, pos)

; Coloration des mots clés JavaScript
Declare colorizeJSKeywords(gadget, startPos, endPos)

; Coloration des mots clés JavaScript
Declare colorizeJSComments(gadget, startPos, endPos)

; Coloration des nomnbres 
Declare colorizeJSNumbers(gadget, startPos, endPos)

; Coloration des string entre guillemets
Declare colorizeJSStrings(gadget, startPos, endPos)

; Coloration des brackets '{' '}' '[' ']'
Declare colorizeJSBrackets(gadget, startPos, endPos)

; Coloration des parenthéses
Declare colorizeJSParentheses(gadget, startPos, endPos) 

; Retourne les positions de début et fin du code CSS
Declare getCSSRange(gadget, *start.Integer, *end.Integer)

; Colorer des mots clés CSS
Declare colorizeCSS(gadget, startPos, endPos)

; Mettre à jour le pliage/dépliage
Declare updateHTMLFolding(gadget, KeywordsFolding.s) 

; Texte au format UTF8
Declare makeUTF8Text(text.s)

; Evenements
Declare onTabBarClick()
Declare onResize()
Declare onClose()
;}

;- Fenetre de l'application
OpenWindow(#app, 0, 0, 800, 600, "HTML Playground " + version, windowStyle)
SetWindowColor(#app, windowColor)
SetGadgetFont(#PB_Default, FontID(font))

; Barre de status
If CreateStatusBar(#StatusBar,WindowID(#app))
  AddStatusBarField(150)
  AddStatusBarField(450) 
EndIf 

; Neutraliser la touche TAB et les caractéres spéciaux
RemoveKeyboardShortcut(#app, #PB_Shortcut_Tab)

AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_B, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_G, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_E, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_R, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_O, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_P, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_Q, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_S, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_F, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_H, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_K, 0)
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_W, 0)  
AddKeyboardShortcut(#app, #PB_Shortcut_Control+#PB_Shortcut_N, 0)

;- Tabbar gadget
TabBarGadgetInclude\TabBarColor = windowColor
TabBarGadget(#TB, 2, 50, WindowWidth(#app)-2, 22, 0, #app)
AddTabBarGadgetItem(#TB, #PB_Default, "HTML")
AddTabBarGadgetItem(#TB, #PB_Default, "Voir le résultat")
SetTabBarGadgetState(#TB, 0) ;Set active panel

SetTabBarGadgetItemData(#TB, 0, 0) 
SetTabBarGadgetItemData(#TB, 1, 1)

; Editeur Scintilla gadget associé à un callback 
ScintillaGadget(#editor, 0, GadgetY(#tb)+GadgetHeight(#tb), 800, WindowHeight(#app)-MarginBot, @scintillaCallBack())

;- Parametres Scintilla (Couleur, font, numerotation, style,  etc ...)
scintillaProperties(#editor)

; Positionner le curseur de saisie dans l'éditeur
SetActiveGadget(#editor)

;- WebView
WebViewGadget(#webView, 0, GadgetY(#tb)+GadgetHeight(#tb), 800, WindowHeight(#app)-MarginBot, #PB_WebView_Debug)
HideGadget(#webView, #True)

; "<!--" et "-->" convertis au format UTF8
PokeS(commentStartPattern, "<!--", -1, #PB_UTF8)
PokeS(commentEndPattern,   "-->",  -1, #PB_UTF8)

PokeS(commentCSSStartPattern, "/*", -1, #PB_UTF8)
PokeS(commentCSSEndPattern,   "*/",  -1, #PB_UTF8)

;- Déclencheurs
BindGadgetEvent(#TB, @OnTabBarClick(), #TabBarGadget_EventType_Change)
BindEvent(#PB_Event_SizeWindow, @onResize(), #app)
BindEvent(#PB_Event_CloseWindow, @onClose(), #app)

; Initialisation HTML (Code par défaut dans le Gadget Scintilla
newProject()

;- Boucle évenementielle
Repeat : WaitWindowEvent(10) : ForEver

;-
;- Les procédures

; Insérer le code HTML par défaut dans le gadget Scintilla
Procedure newProject()
  Protected size, *memoryId
  
  With currentProject
    \name   = "New Project.json"
    \html   = "<!DOCTYPE html>" + #CRLF$ +
              "<html>" + #CRLF$ +
              "<head>" + #CRLF$ +
              "<title>Page Title</title>" + #CRLF$ +
              "</head>" + #CRLF$ +
              "<body>" + #CRLF$ +
              "    <h1>Hooo le joli titre 🌞</h1>" + #CRLF$ +
              "</body>" + #CRLF$ + 
              "</html>" 
  EndWith  
  
  ; Insérer le code HTML dans le gadget Scintilla
  size = StringByteLength(currentProject\html, #PB_UTF8) + 1
  *memoryID = AllocateMemory(size)
  PokeS(*MemoryID, currentProject\html, -1, #PB_UTF8)
  
  ScintillaSendMessage(#editor, #SCI_SETTEXT, 0, *memoryID)
  FreeMemory(*MemoryID)
EndProcedure

; Parametres du gadget scintilla
Procedure scintillaProperties(gadget)
  ;- 1 Style par défaut
  ; Couleur et police de caractéres
  ScintillaSendMessage(gadget, #SCI_STYLESETFORE, #STYLE_DEFAULT, scFrontColor)
  ScintillaSendMessage(gadget, #SCI_STYLESETBACK, #STYLE_DEFAULT, scBackColor)
  
  ScintillaSendMessage(gadget, #SCI_STYLESETFONT,#STYLE_DEFAULT, @scFont) 
  ScintillaSendMessage(gadget, #SCI_STYLESETSIZE, #STYLE_DEFAULT, scFontSize)
  
  ;- 2 Propager vers tous les styles
  ScintillaSendMessage(gadget, #SCI_STYLECLEARALL)
  
  ;- 3 Définir des styles spécifiques du gadget Scintilla 
  ; Activation et couleur de la ligne en cours d'édition
  ScintillaSendMessage(gadget, #SCI_SETCARETLINEVISIBLE, #True)
  ScintillaSendMessage(gadget, #SCI_SETCARETLINEBACK, scCurrentLineColor)
  
  ; Les tabulations sont remplacées par des espaces 
  ScintillaSendMessage(gadget, #SCI_SETUSETABS, #False)
  
  ; Nombre d'espaces pour une tabulation
  ScintillaSendMessage(gadget, #SCI_SETINDENT, 4)
  
  ; Création de la colonne de numérotation des lignes
  ScintillaSendMessage(gadget, #SCI_SETMARGINTYPEN, 1, #SC_MARGIN_NUMBER) 
  ScintillaSendMessage(gadget, #SCI_SETMARGINWIDTHN, 1, scMarginSizeDefault)
  
  ; Autocomplétion : Parametres de la liste des mots clés 
  ScintillaSendMessage(gadget, #SCI_AUTOCSETMAXHEIGHT, 20)
  ScintillaSendMessage(gadget, #SCI_AUTOCSETMAXWIDTH, 150)
  ScintillaSendMessage(gadget, #SCI_AUTOCSETAUTOHIDE, #True)
  ScintillaSendMessage(gadget, #SCI_AUTOCSETCHOOSESINGLE, #True)
  ScintillaSendMessage(gadget, #SCI_AUTOCSETIGNORECASE, #True)
  
  ; Autocomplétion : Caractére séparant chaque mot de la liste des mots clés
  ScintillaSendMessage(gadget, #SCI_AUTOCSETSEPARATOR, Asc(KeyWordSep))
  
  ; Autocomplétion : Caractére sélectionnant le mot de la liste d'autocomplétion
  ScintillaSendMessage(gadget, #SCI_AUTOCSETFILLUPS, 0, @" ")
  
  ; Autocomplétion : Tri de la liste 
  ScintillaSendMessage(gadget, #SCI_AUTOCSETORDER, #SC_ORDER_PERFORMSORT) 
  
  ; Folding : Création de la colonne de pliages/dépliage
  ScintillaSendMessage(gadget, #SCI_SETMARGINMASKN, 2, #SC_MASK_FOLDERS)
  ScintillaSendMessage(gadget, #SCI_SETMARGINWIDTHN, 2, 20)
  ScintillaSendMessage(gadget, #SCI_SETMARGINSENSITIVEN, 2, #True)
  
  ; Folding : Choix des icones de pliages du code 5
  ScintillaSendMessage(gadget, #SCI_MARKERDEFINE, #SC_MARKNUM_FOLDEROPEN, #SC_MARK_CIRCLEMINUS) 
  ScintillaSendMessage(gadget, #SCI_MARKERDEFINE, #SC_MARKNUM_FOLDER, #SC_MARK_CIRCLEPLUS)
  ScintillaSendMessage(gadget, #SCI_MARKERDEFINE, #SC_MARKNUM_FOLDERSUB, #SC_MARK_VLINE) ;Ligne verticale
  ScintillaSendMessage(gadget, #SCI_MARKERDEFINE, #SC_MARKNUM_FOLDERTAIL, #SC_MARK_LCORNERCURVE) ;fin arborescence
  ScintillaSendMessage(gadget, #SCI_MARKERDEFINE, #SC_MARKNUM_FOLDEREND, #SC_MARK_CIRCLEPLUSCONNECTED) ;Dépliage
  ScintillaSendMessage(gadget, #SCI_MARKERDEFINE, #SC_MARKNUM_FOLDEROPENMID, #SC_MARK_CIRCLEMINUSCONNECTED) ;Pliage
  ScintillaSendMessage(gadget, #SCI_MARKERDEFINE, #SC_MARKNUM_FOLDERMIDTAIL, #SC_MARK_TCORNERCURVE)
  
  ; Folding : Couleur des icones de pliages/dépliage
  ScintillaSendMessage(gadget, #SCI_MARKERSETFORE, #SC_MARKNUM_FOLDER, scMasterFolder)
  ScintillaSendMessage(gadget, #SCI_MARKERSETBACK, #SC_MARKNUM_FOLDER, RGB(0, 0, 0))
  
  ScintillaSendMessage(gadget, #SCI_MARKERSETFORE, #SC_MARKNUM_FOLDEROPEN, scMasterFolder)
  ScintillaSendMessage(gadget, #SCI_MARKERSETBACK, #SC_MARKNUM_FOLDEROPEN, RGB(0, 0, 0))
  
  ScintillaSendMessage(gadget, #SCI_MARKERSETBACK, #SC_MARKNUM_FOLDEROPENMID, RGB(0, 0, 0))
  ScintillaSendMessage(gadget, #SCI_MARKERSETBACK, #SC_MARKNUM_FOLDERSUB, RGB(0, 0, 0))
  
  ScintillaSendMessage(gadget, #SCI_MARKERSETBACK, #SC_MARKNUM_FOLDERTAIL, RGB(0, 0, 0))
  ScintillaSendMessage(gadget, #SCI_MARKERSETBACK, #SC_MARKNUM_FOLDERMIDTAIL, RGB(0, 0, 0))
  
  ;- Folding : Activation du folding
  ScintillaSendMessage(gadget, #SCI_SETMARGINTYPEN, 2, #SC_MARGIN_SYMBOL)
  ScintillaSendMessage(gadget, #SCI_SETMARGINMASKN, 2, #SC_MASK_FOLDERS)
  ScintillaSendMessage(gadget, #SCI_SETMARGINSENSITIVEN, 2, 1)
  ScintillaSendMessage(gadget, #SCI_SETMARGINWIDTHN, 2, 20)
  
  
  ;- Définir les styles HTML/CSS/JS
  
  ; Styles : Coloration des mots clés HTML 
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_HTML_Keyword, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_HTML_Keyword, scStyleHTMLKeyWordColor)
  
  ; Style commentaires HTML
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_HTML_Comment, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_HTML_Comment, scStyleHTMLCommentColor)
  
  ; Styles : Coloration des commentaire JavaScript
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_JS_Comment, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_JS_Comment, scStyleJSCommentColor)
  
  
  ; Styles : Coloration des commentaire JavaScript
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_JS_Comment, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_JS_Comment, scStyleJSCommentColor)
  
  ; Styles : Coloration des mots clés JavaScript
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_JS_KeyWord, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_JS_KeyWord, scStyleJSKeywordColor)
  
  ; Styles : Coloration des Nombres JavaScript
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_JS_Number, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_JS_Number, scStyleJSNumber)
  
  ; Styles : Strings JS 
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_JS_String, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_JS_String, scStyleJSString)
  
  ; Styles : Brackets JS 
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_JS_Bracket, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_JS_Bracket, scStyleJSSBracket)
  
  ; Styles : Parentheses JS 
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_JS_Parentheses, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_JS_Parentheses, scStyleJSParentheses)
  
  ; Styles : Sélecteurs CSS
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_CSS_Selector, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_CSS_Selector, scStyleCSSKeywordColor)
  
  ; Syles : Propriétés CSS
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_CSS_Property, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_CSS_Property, scStyleCSSProperty)
  
  ; Styles : Valeurs CSS
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_CSS_Value, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_CSS_Value, scStyleCSSValue)
  
  ; Styles : Nombres CSS
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_CSS_Number, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_CSS_Number, scStyleCSSNumber)
  
  ; Styles : Strings CSS
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_CSS_String, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_CSS_String, scStyleCSSString)
  
  ; Styles : Commentaires CSS
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_CSS_Comment, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_CSS_Comment, scStyleCSSComment)
  
  ; Styles : Ponctuation CSS
  ScintillaSendMessage(gadget, #SCI_INDICSETSTYLE, #Style_CSS_Punct, #INDIC_TEXTFORE)
  ScintillaSendMessage(gadget, #SCI_INDICSETFORE,  #Style_CSS_Punct, scStyleCSSPunct) 
EndProcedure

;-

; Callback evenementielle du gadget scintilla
Procedure scintillaCallBack(gadget, *scn.scNotification)
  Protected textLength
  Protected scCurrentPos, SciWordStartPos
  Protected curPos, curPos2
  Protected *buf, *buf2
  Protected tag.s   ; Rechercher le tag entre "<" et ">" pour auto-fermeture du tag 
  Protected close.s ; Création du tag de fermeture "</tag>
  
  Protected jsCurrentPos
  Protected cssStart, cssEnd ; Ou commence et se termine la déclaration du code CSS
  Protected jsStart, jsEnd   ; Ou commence et se termine la déclaration du code Style
  
  ; Calculer la longueur du texte contenu dans le gadget scintilla
  textLength = ScintillaSendMessage(gadget, #SCI_GETLENGTH)
  
  ; Analyser les evenement scintilla
  Select *scn\nmhdr\code
    Case #SCN_CHARADDED 
      ; Un caractére est saisie
      tabPreviousId = GetTabBarGadgetItemPosition(#tb, #TabBarGadgetItem_Selected)
      
      Select *scn\ch
        Case 13
          ;- Indentation
          ; Vous avez pressé la touche entrée
          
          ; Quel est la position du décalages du texte au niveau de la marge.
          scIndent = ScintillaSendMessage(gadget, #SCI_GETLINEINDENTATION, scLine)
          
          ; Passer à la ligne suivante 
          ; et positionner le curseur 
          ScintillaSendMessage(gadget, #SCI_SETLINEINDENTATION, scLine+1, scIndent)
          
          ; Touche entrée préssée : Le curseur ne passera pas à  la ligne suivante.
          ; Forcer le passage à la ligne en insérant la longueur de #CRLF$
          If scIndent=0 
            scPos = scPos + Len(#CRLF$)
          EndIf
          
          ; Positionner le curseur de saisie 
          ScintillaSendMessage(gadget, #SCI_GOTOPOS, scPos + scIndent)
          
        Case 'a' To 'z', 'A' To 'Z'
          ;- Autocomplétion (Affichage d'une liste contenant les mots clés HTML
          ;  Affichage du mot selectionné si autocomplétion
          scCurrentPos = ScintillaSendMessage(gadget, #SCI_GETCURRENTPOS)
          SciWordStartPos = ScintillaSendMessage(gadget, #SCI_WORDSTARTPOSITION, scCurrentPos, 1)
          ScintillaSendMessage(gadget, #SCI_AUTOCSHOW, scCurrentPos - SciWordStartPos, makeUTF8Text(htmlKeyWords))
          
        Case  '>'
          ;- Auto fermeture de tag <tag>|</tag>
          
          ; Obtenir le tag de fermeture
          tag = getOpeningTagBeforeCaret(gadget)
          
          If tag <> ""
            
            curPos2 = ScintillaSendMessage(gadget, #SCI_GETCURRENTPOS, 0, 0)
            close = "</" + tag + ">"
            
            *buf2 = AllocateMemory(StringByteLength(close, #PB_UTF8) + 1)
            PokeS(*buf2, close, -1, #PB_UTF8)
            
            ScintillaSendMessage(gadget, #SCI_INSERTTEXT, curPos2, *buf2)
            FreeMemory(*buf2)
            
            ; On ne modifie pas la position du curseur.
            ; Il est positionné entre le tag d'ouverture et le tag de fermeture
          EndIf      
      EndSelect
      
    Case #SCN_UPDATEUI        
      ; Une mise à jour du gadget Scintilla est effectué 
      
      ; Style & folding UNIQUEMENT si le contenu change
      If *scn\updated & #SC_UPDATE_CONTENT And textLength > 0
        
        ; Sauvegarde du code à chaque modification de celui-ci
        currentProject\html = GetGadgetText(#editor)
        
        ;- Coloration HTML et commentaire
        colorizeHTMLTags(gadget, 0, textLength, htmlKeyWords)
        
        ;- Coloration Javascript - Etre entre les balise <script> et </script> 
        ;  Il peut il y avoir plusieurs script JavaScript
        While getJSRange(gadget, @jsStart, @jsEnd, jsCurrentPos)          
          If jsStart <> 0 
            colorizeJSStrings(gadget, jsStart, jsEnd)
            colorizeJSComments(gadget, jsStart, jsEnd) 
            colorizeJSNumbers(gadget, jsStart, jsEnd)
            colorizeJSKeywords(gadget, jsStart, jsEnd)
            colorizeJSBrackets(gadget, jsStart, jsEnd)
            colorizeJSParentheses(gadget, jsStart, jsEnd)
          EndIf
          jsCurrentPos = jsEnd + Len("</scipt>")
        Wend
        
        ;- Coloration CSS - Etre entre les balises <style> et </style>
        getCSSRange(gadget, @cssStart, @cssEnd)       
        If cssStart <> -1 And cssEnd > cssStart
          colorizeCSS(gadget, cssStart, cssEnd)
        EndIf
        
        ; Mise à jours des flags de folding (Pliage / dépliage de code)
        updateHTMLFolding(gadget, KeywordsFolding)
      EndIf
      
    Case #SCN_MARGINCLICK
      ; Mise en oeuvre du pliage dépliage (folding)
      
      ; Clique sur la deuxiéme colonne
      If *scn\margin = 2
        
        ; Ligne réelle à partir de la position du clic
        Protected line = ScintillaSendMessage(gadget, #SCI_LINEFROMPOSITION, *scn\position)
        Protected level = ScintillaSendMessage(gadget, #SCI_GETFOLDLEVEL, line)
        
        ; Autoriser uniquement sur les lignes HEADER
        If level & #SC_FOLDLEVELHEADERFLAG
          ScintillaSendMessage(gadget, #SCI_TOGGLEFOLD, line)
        EndIf
      EndIf      
  EndSelect   
  
  ; Important pour le fonctionnement de l'indentation
  
  ; Trouver la position du curseur de saisie  
  scPos = ScintillaSendMessage(gadget, #SCI_GETANCHOR)
  
  ; Trouver la ligne ou se touve le curseur
  scLine = ScintillaSendMessage(gadget, #SCI_LINEFROMPOSITION, scPos)
  
  ; Trouver la colonne en cours
  scCol = ScintillaSendMessage(gadget, #SCI_GETCOLUMN, scPos)
  
  ; Affichage du numero de ligne/colonne dans la barre de status
  StatusBarText(#StatusBar, 0, "Line : " +Str(scLine+1)+ "  Col : "+Str(scCol+1), #PB_StatusBar_Center)  
  
EndProcedure

;-

; Obtenir le texte de la ligne en cours de saisie 
Procedure.s getScintillaLineText(gadget, line)
  Protected lineLength, *buffer, result.s
  
  ; Longueur de la ligne
  lineLength = ScintillaSendMessage(gadget, #SCI_LINELENGTH, line)
  
  ; Initialisation du buffer UTF8
  *buffer = AllocateMemory(lineLength + 1)
  
  If *buffer > 0
    ; Obtenir et renvoyer le contenu de la ligne
    ScintillaSendMessage(gadget, #SCI_GETLINE, line, *buffer)
    result.s = PeekS(*buffer, -1, #PB_UTF8)
    FreeMemory(*buffer)
  EndIf   
  ProcedureReturn result
EndProcedure

; Obtenir le texte se trouvant entre deux positions
Procedure.s getScintillaWord(gadget, startPos, endPos)
  Protected tr.scTextRange, *buffer, length, result.s
  
  length = endPos - startPos
  
  If length > 0
    *buffer = AllocateMemory(length + 1)
    
    tr\chrg\cpMin = startPos
    tr\chrg\cpMax = endPos
    tr\lpstrText  = *buffer
    
    ScintillaSendMessage(gadget, #SCI_GETTEXTRANGE, 0, @tr)
    
    ; Lecture UTF-8 correcte
    result = PeekS(*buffer, -1, #PB_UTF8)
    FreeMemory(*buffer)
  EndIf 
  
  ProcedureReturn result
EndProcedure

; Tester si un mot est un mot clé
Procedure.b isKeyword(word.s, keyWords.s)
  Protected countWords = CountString(keyWords, KeyWordSep), n
  Protected result.b  
  
  For n = 1 To countWords + 1
    If LCase(StringField(keyWords, n, KeyWordSep)) = LCase(word)
      result = #True
    EndIf
  Next
  ProcedureReturn result
EndProcedure

; Rechercher le tag précédent le curseur de saisie
Procedure.s getOpeningTagBeforeCaret(gadget)
  Protected curPos, startPos, endPos, text.s, name.s, i, ch
  
  ; Déterminer la position du curseur
  curPos = ScintillaSendMessage(gadget, #SCI_GETCURRENTPOS)
  startPos = curPos - 2   ; juste avant '>'
  
  If startPos < 0 : ProcedureReturn "" : EndIf
  
  ; Remonter jusqu'à '<'
  For i = startPos To 0 Step -1
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, i)
    If ch = '<'
      startPos = i
      Break
    EndIf
  Next
  
  If ch <> '<' : ProcedureReturn "" : EndIf
  
  ; lire le contenu <...>
  endPos = curPos - 1
  text = GetScintillaWord(gadget, startPos, endPos)
  text = Trim(text)
  
  ; Rejeter </tag> et <br/>
  If Left(text, 2) = "</" : ProcedureReturn "" : EndIf
  If Right(text, 2) = "/>" : ProcedureReturn "" : EndIf
  
  ; Extraire nom après '<'
  For i = 2 To Len(text)
    ch = Asc(Mid(text, i, 1))
    If (ch >= 'a' And ch <= 'z') Or (ch >= 'A' And ch <= 'Z') Or
       (ch >= '0' And ch <= '9') Or ch = '-' Or ch = '_'
      name + Chr(ch)
    Else
      Break
    EndIf
  Next
  
  ProcedureReturn name
EndProcedure


; Tester si un mot trouvé dans un string déclenche le folding
Procedure.b isFoldingKeyWord(text.s, keyWordsfolding.s)
  Protected countWords, n
  Protected result.b  
  
  ; Compter le nombre de mots clés folding contenu dans la variable KeyWordsFolding 
  countWords = CountString(keyWordsfolding, KeyWordSep)
  
  ; Pour chaque mots clé folding
  For n = 1 To countWords + 1
    ; Vérifier que le mot clé se trouve dans dans la chaine reçu
    ; Il doit se trouver au début (Position 1) de la chaine testé (text.s)
    If FindString(LCase(text), LCase(StringField(keyWordsfolding, n, KeyWordSep))) = 1   
      result = #True
    EndIf
  Next
  
  ProcedureReturn result
EndProcedure


;-

; Coloration des mots clés HTML
Procedure colorizeHTMLTags(gadget, startPos, endPos, keyWords.s)
  Protected pos                       ; Position actuelle de l'indice de parcours du string
  Protected lowerTag                  ; Position du prochain "<"
  Protected upperTag                  ; Position du prochain ">"
  Protected tagStart                  ; Premier caractère après "<"
  Protected tagEnd                    ; Fin du nom de la balise sans ">"
  Protected originalTagStart          ; Début logique de la balise (après "<")
  Protected tagName.s                 ; Tag sans balise d'ouverture/fermeture "<tag> -> "tag"
  Protected nameStart                 ; Premier caractère après "<" 
  Protected isClosing                 ; Indicateur de balise fermée
  Protected space                     ; Position d'un espace à l'intérieur de "<" et ">"
  Protected commentStart, commentEnd  ;Position de début et fin d'un commentaire
  
  ; La coloration se fera en plusieurs passage 
  ; - 1 Commentaires
  ; - 2 Tags HTML <tag> </tag>
  
  ;- ■ 1 Pass commentaires <!-- Commentaire -->
  pos = startPos
  
  ; Reset des indicateurs de styles
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_HTML_Comment, 0)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  While pos < endPos
    
    ; Définir le target de recherche
    ScintillaSendMessage(gadget, #SCI_SETTARGETSTART, pos)
    ScintillaSendMessage(gadget, #SCI_SETTARGETEND, endPos)
    
    ; Rechercher la chaine de début de commentaire "<!--" (Longueur 4)
    commentStart = ScintillaSendMessage(gadget, #SCI_SEARCHINTARGET, 4, commentStartPattern)
    
    ; Il y en a pas, on s'arrete !
    If commentStart = -1 : Break : EndIf
    
    ; Le début du commentaire est trouvé
    ; Nouveau target de recherche qui commencera aprés "<!--"
    ScintillaSendMessage(gadget, #SCI_SETTARGETSTART, commentStart + 4)
    ScintillaSendMessage(gadget, #SCI_SETTARGETEND, endPos)
    
    ; Rechercher la chaine de fin de commentaire "-->" (Longueur 3)
    commentEnd = ScintillaSendMessage(gadget, #SCI_SEARCHINTARGET, 3, commentEndPattern)
    
    If commentEnd <> -1
      commentEnd + 3
    Else
      commentEnd = endPos
    EndIf
    
    ; colorer le commentaire
    ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_HTML_Comment)
    ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, commentStart, commentEnd - commentStart)
    
    pos = commentEnd
  Wend
  
  ;- ■ 2 Pass tag HTML "<html>"
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_HTML_Keyword, 0)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  pos = startPos
  
  While pos < endPos
    
    ; Définir le target de recherche
    ScintillaSendMessage(gadget, #SCI_SETTARGETSTART, pos)
    ScintillaSendMessage(gadget, #SCI_SETTARGETEND, endPos)
    
    ; Chercher le caractére '<' dans la zone de scan 
    lowerTag = ScintillaSendMessage(gadget, #SCI_SEARCHINTARGET, 1, @"<")
    
    ; Il y en a pas. On sort de la boucle
    If lowerTag = -1 : Break : EndIf
    
    ; "<" est trouvé
    ; Nouveau target de recherche qui commencera aprés "<"
    ScintillaSendMessage(gadget, #SCI_SETTARGETSTART, lowerTag + 1) 
    ScintillaSendMessage(gadget, #SCI_SETTARGETEND, endPos)
    
    ; Chercher le prochain '>'
    upperTag = ScintillaSendMessage(gadget, #SCI_SEARCHINTARGET, 1, @">")
    
    If upperTag = -1
      ; ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_HTML_Keyword, 0)
      ; ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, lowerTag, 1)
      Break
    EndIf
    
    ; Bonne nouvelle nous avons maintenant localisé "<" et ">"
    originalTagStart = lowerTag + 1
    tagStart         = originalTagStart
    
    ; Nouveau target de recherche qui commencera aprés "<" et se termine à ">"
    ScintillaSendMessage(gadget, #SCI_SETTARGETSTART, tagStart)
    ScintillaSendMessage(gadget, #SCI_SETTARGETEND, upperTag)
    
    ; Recherche du caractéres espace dans la chaine située entre "<" et ">" 
    space = ScintillaSendMessage(gadget, #SCI_SEARCHINTARGET, 1, @" ")
    
    If space = -1
      tagEnd = upperTag
    Else
      tagEnd = space
    EndIf
    
    ; Rechercher le mot situé entre "<" et ">" 
    If tagEnd > tagStart
      tagName = GetScintillaWord(gadget, tagStart, tagEnd)
      isClosing = #False
      nameStart = tagStart
      
      ; Détecter </tag>
      If Left(tagName, 1) = "/"
        isClosing = #True
        tagName  = Mid(tagName, 2)
        nameStart = tagStart + 1
      EndIf
      
      ; Indiquer le début de la coloration  style #Style_HTML_Keyword 
      ; Indiquer la position de débug et fin de la coloration
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_HTML_Keyword)
      ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, originalTagStart, upperTag - originalTagStart)
      
      ; Coloration de la balise "<" - position de début sur une longueur de 1
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, lowerTag, 1)
      
      ; Coloration de "/" si balise fermante - position de début sur une longueur de 1
      If isClosing
        ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, tagStart, 1)
      EndIf
      
      ; Coloration du nom de balise (ouvrante OU fermante) - Exemple html
      If IsKeyword(tagName, keyWords)
        ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, nameStart, tagEnd - nameStart)
      EndIf
      
      ; Coloration de ">" - position de fin sur une longueur de 1
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, upperTag, 1)
    EndIf
    
    ; AU suivant ...
    pos = upperTag + 1
  Wend  
EndProcedure

;-
;- Coloration JavaScript

; Retourne les positions de début et fin du code JavaScript
Procedure getJSRange(gadget, *start.Integer, *end.Integer, currentPos)
  Protected textLen = ScintillaSendMessage(gadget, #SCI_GETLENGTH, 0, 0)
  Protected pos = currentPos
  Protected inScript = #False
  Protected jsStart = -1
  Protected jsEnd   = -1
  Protected word.s
  
  While pos < textLen
    If ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0) = '<'
      
      word = LCase(GetScintillaWord(gadget, pos + 1, pos + 20))
      
      If FindString(word, "script") And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) <> '/'
        inScript = #True
        jsStart = pos
        
        ; avancer après >
        While pos < textLen And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0) <> '>'
          pos + 1
        Wend
        
        jsStart = pos + 1
      EndIf
      
      If FindString(word, "/script")
        jsEnd = pos
        
        *start\i = jsStart
        *end\i   = jsEnd
        ProcedureReturn
      EndIf
    EndIf
    
    pos + 1
  Wend
EndProcedure


; Chercher la fin d'un commentaire JavasCript
Procedure skipJSLineComment(gadget, pos)
  Protected ch
  
  While pos < ScintillaSendMessage(gadget, #SCI_GETLENGTH, 0, 0)
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    If ch = 10 Or ch = 13
      Break
    EndIf
    pos + 1
  Wend
  ProcedureReturn pos
EndProcedure

; Ignorer les blocks de commentaires JavaScript "/* ... */"
Procedure skipJSBlockComment(gadget, pos)
  pos + 2
  While pos < ScintillaSendMessage(gadget, #SCI_GETLENGTH, 0, 0) - 1
    If ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0) = '*' And
       ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '/'
      pos + 2
      Break
    EndIf
    pos + 1
  Wend
  ProcedureReturn pos
EndProcedure


; Colorer les mots clés JavaScript 
Procedure colorizeJSKeywords(gadget, startPos, endPos)
  Protected pos = startPos
  Protected wordStart, wordEnd, word.s
  Protected ch
  
  ; Reset de la coloration des mots clés JavaScript
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_KeyWord)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  While pos < endPos
    
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    
    ; Ignorer les lignes de commentaires JavaScript "//"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '/'
      pos = skipJSLineComment(gadget, pos)
      Continue
    EndIf 
    
    ; Ignorer les lignes de bloc de commentaires JavaScript "/* ... */"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '*'
      pos = skipJSBlockComment(gadget, pos)
      Continue
    EndIf
    
    ; Sauter non-lettres    
    If (ch < 'A' Or ch > 'Z') And (ch < 'a' Or ch > 'z') And ch <> '_'
      pos + 1
      Continue
    EndIf
    
    ; Lire mot
    wordStart = pos
    
    While pos < endPos
      ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
      If (ch >= 'A' And ch <= 'Z') Or (ch >= 'a' And ch <= 'z') Or 
         (ch >= '0' And ch <= '9') Or ch = '_'
        pos + 1
      Else
        Break
      EndIf
    Wend
    
    wordEnd = pos
    word = GetScintillaWord(gadget, wordStart, wordEnd)
    
    If IsKeyword(word, jsKeywords)
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_KeyWord, 0)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, wordStart, wordEnd - wordStart)
    EndIf
    
  Wend
EndProcedure

Procedure colorizeJSComments(gadget, startPos, endPos)
  Protected pos = startPos
  Protected ch, nextCh
  Protected commentStart, commentEnd
  
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Comment)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  While pos < endPos - 1
    
    ch     = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    nextCh = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0)
    
    ; // commentaire ligne
    If ch = '/' And nextCh = '/'
      
      commentStart = pos
      
      ; Aller jusqu'à fin de ligne ou fin zone
      pos + 2
      While pos < endPos
        ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
        If ch = 10 Or ch = 13   ; LF ou CR
          Break
        EndIf
        pos + 1
      Wend
      
      commentEnd = pos
      
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Comment, 0)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, commentStart, commentEnd - commentStart)
      
      Continue
    EndIf
    
    ; /* commentaire bloc */
    If ch = '/' And nextCh = '*'
      
      commentStart = pos
      
      pos + 2
      While pos < endPos - 1
        ch     = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
        nextCh = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0)
        If ch = '*' And nextCh = '/'
          pos + 2
          Break
        EndIf
        pos + 1
      Wend
      
      commentEnd = pos
      
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Comment)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, commentStart, commentEnd - commentStart)
      
      Continue
    EndIf
    
    pos + 1
  Wend
EndProcedure

; Coloration des chiffres dans un script Javascript
Procedure colorizeJSNumbers(gadget, startPos, endPos)
  
  Protected pos = startPos
  Protected ch, nextCh
  Protected numberStart
  Protected hasDot, hasExp
  
  ; Reset du style
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Number)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  While pos < endPos
    
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    
    ; Ignorer les lignes de commentaires JavaScript "//"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '/'
      pos = skipJSLineComment(gadget, pos)
      Continue
    EndIf 
    
    ; Ignorer les lignes de bloc de commentaires JavaScript "/* ... */"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '*'
      pos = skipJSBlockComment(gadget, pos)
      Continue
    EndIf
    
    ; Début possible de nombre
    If (ch >= '0' And ch <= '9') Or
       (ch = '.' And (ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) >= '0' And
                      ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) <= '9'))
      
      numberStart = pos
      hasDot = #False
      hasExp = #False
      
      ; hex / bin / oct
      If ch = '0'
        nextCh = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0)
        If nextCh = 'x' Or nextCh = 'X' Or nextCh = 'b' Or nextCh = 'B' Or nextCh = 'o' Or nextCh = 'O'
          pos + 2
          While pos < endPos
            ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
            If (ch >= '0' And ch <= '9') Or
               (ch >= 'a' And ch <= 'f') Or
               (ch >= 'A' And ch <= 'F')
              pos + 1
            Else
              Break
            EndIf
          Wend
          
          ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Number, 0)
          ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, numberStart, pos - numberStart)
          
          Continue
        EndIf
      EndIf
      
      ; Décimal / flottant / scientifique
      While pos < endPos
        
        ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
        
        If ch >= '0' And ch <= '9'
          pos + 1
          Continue
        EndIf
        
        If ch = '.' And hasDot = #False
          hasDot = #True
          pos + 1
          Continue
        EndIf
        
        If (ch = 'e' Or ch = 'E') And hasExp = #False
          hasExp = #True
          pos + 1
          
          ; Signe optionnel après e
          ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
          If ch = '+' Or ch = '-'
            pos + 1
          EndIf
          
          Continue
        EndIf
        
        Break
      Wend
      
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Number, 0)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, numberStart, pos - numberStart)
      
      Continue
    EndIf
    
    pos + 1
  Wend
EndProcedure

; Coloration des string situés dans un code JavaScript
Procedure colorizeJSStrings(gadget, startPos, endPos)
  Protected pos = startPos
  Protected ch, quote
  Protected stringStart, stringEnd
  Protected escaped
  
  ; Reset du style
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_String, 0)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  While pos < endPos
    
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    
    ; Ignorer les lignes de commentaires JavaScript "//"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '/'
      pos = skipJSLineComment(gadget, pos)
      Continue
    EndIf 
    
    ; Ignorer les lignes de bloc de commentaires JavaScript "/* ... */"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '*'
      pos = skipJSBlockComment(gadget, pos)
      Continue
    EndIf
    
    ; Détecter début des string : caractére " ou '  `
    If ch = 34 Or ch = 39 Or ch = 96
      quote = ch
      stringStart = pos
      pos + 1
      escaped = #False
      
      While pos < endPos
        
        ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
        
        If escaped
          escaped = #False
          pos + 1
          Continue
        EndIf
        
        If ch = 92   ; backslash \
          escaped = #True
          pos + 1
          Continue
        EndIf
        
        ; fermeture string
        If ch = quote
          pos + 1
          Break
        EndIf
        
        pos + 1
      Wend
      
      stringEnd = pos
      
      ; Coloration du string 
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_String, 0)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, stringStart, stringEnd - stringStart)
      
      Continue
    EndIf
    
    pos + 1
  Wend  
EndProcedure

; Coloration des brackets '{' '}' '[' ']' dans un script JavaScript
Procedure colorizeJSBrackets(gadget, startPos, endPos)
  Protected pos, ch
  
  ; Reset du style
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Bracket)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  
  For pos = startPos To endPos - 1
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    
    If ch = '{' Or ch = '}' Or ch = '[' Or ch = ']'
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Bracket)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, pos, 1)
    EndIf
  Next
EndProcedure

; Coloration des parenthéses dans un script JavaScript
Procedure colorizeJSParentheses(gadget, startPos, endPos)  
  Protected pos = startPos
  Protected ch
  
  ; Reset du style
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Parentheses)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  While pos < endPos
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    
    ; Ignorer les lignes de commentaires JavaScript "//"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '/'
      pos = skipJSLineComment(gadget, pos)
      Continue
    EndIf 
    
    ; Ignorer les lignes de bloc de commentaires JavaScript "/* ... */"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '*'
      pos = skipJSBlockComment(gadget, pos)
      Continue
    EndIf
    
    If ch = '(' Or ch = ')'
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_JS_Parentheses)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, pos, 1)
    EndIf
    
    pos + 1
  Wend
EndProcedure

;-

;- Coloration CSS

; Retourne les positions de début et fin du code CSS
Procedure getCSSRange(gadget, *start.Integer, *end.Integer)
  Protected textLen = ScintillaSendMessage(gadget, #SCI_GETLENGTH, 0, 0)
  Protected pos = 0
  Protected inStyle = #False
  Protected cssStart = -1
  Protected cssEnd   = -1
  Protected word.s
  
  While pos < textLen
    If ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0) = '<'
      
      word = LCase(GetScintillaWord(gadget, pos + 1, pos + 20))
      
      If FindString(word, "style") And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) <> '/'
        inStyle = #True
        cssStart = pos
        
        ; avancer après >
        While pos < textLen And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0) <> '>'
          pos + 1
        Wend
        
        cssStart = pos + 1
      EndIf
      
      If FindString(word, "/style")
        cssEnd = pos
        
        *start\i = cssStart
        *end\i   = cssEnd
        ProcedureReturn
      EndIf
    EndIf
    
    pos + 1
  Wend
  
  *start\i = -1
  *end\i   = -1
EndProcedure


; Colorer les mots clés CSS
Procedure colorizeCSS(gadget, startPos, endPos)
  Protected pos = startPos
  Protected ch, start, word.s
  Protected inSelector = #True
  Protected inBlock = #False
  Protected commentEnd
  Protected quote
  
  ; Reset du style
  ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_Comment, 0)
  ScintillaSendMessage(gadget, #SCI_INDICATORCLEARRANGE, startPos, endPos - startPos)
  
  While pos < endPos
    
    ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
    
    ; Commentaire CSS
    
    ; Recherche début de commentaire CSS "/*"
    If ch = '/' And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos + 1, 0) = '*'      
      ScintillaSendMessage(gadget, #SCI_SETTARGETSTART, pos + 2)
      ScintillaSendMessage(gadget, #SCI_SETTARGETEND, endPos)
      
      ; Recherche fin de commentaire CSS "*/
      commentEnd = ScintillaSendMessage(gadget, #SCI_SEARCHINTARGET, 2, commentCSSEndPattern)
      
      If commentEnd = -1
        commentEnd = endPos
      Else
        commentEnd + 2
      EndIf
      
      ; Invocation du style commentaire et coloration
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_Comment)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, pos, commentEnd - pos)
      
      pos = commentEnd
      Continue
    EndIf
    
    ; String CSS
    If ch = 34 Or ch = 39   ; tester " ou '
      quote = ch
      start = pos
      pos + 1
      
      While pos < endPos
        ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
        
        ; fermeture non échappée
        If ch = quote And ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos - 1, 0) <> 92
          pos + 1
          Break
        EndIf
        
        pos + 1
      Wend
      
      ; Invocation du style sting et coloration
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_String, 0)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, start, pos - start)
      Continue
    EndIf
    
    
    ; Accolades / ponctuation
    If ch = '{' Or ch = '}' Or ch = ':' Or ch = ';'
      
      ; Invocation du style punctuation et coloration
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_Punct, 0)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, pos, 1)
      
      If ch = '{'
        inBlock = #True
        inSelector = #False
      ElseIf ch = '}'
        inBlock = #False
        inSelector = #True
      EndIf
      
      pos + 1
      Continue
    EndIf
    
    ; Nombre CSS
    If (ch >= '0' And ch <= '9')
      
      start = pos
      pos + 1
      
      While pos < endPos
        ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
        If Not ((ch >= '0' And ch <= '9') Or ch = '.' Or ch = '%' Or (ch >= 'a' And ch <= 'z'))
          Break
        EndIf
        pos + 1
      Wend
      
      ; Invocation du style nombre et coloration
      ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_Number, 0)
      ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, start, pos - start)
      Continue
    EndIf
    
    ; Identifiant CSS
    If (ch >= 'a' And ch <= 'z') Or (ch >= 'A' And ch <= 'Z') Or ch = '.' Or ch = '#'
      
      start = pos
      pos + 1
      
      While pos < endPos
        ch = ScintillaSendMessage(gadget, #SCI_GETCHARAT, pos, 0)
        If Not ((ch >= 'a' And ch <= 'z') Or (ch >= 'A' And ch <= 'Z') Or ch = '-' Or (ch >= '0' And ch <= '9'))
          Break
        EndIf
        pos + 1
      Wend
      
      word = LCase(GetScintillaWord(gadget, start, pos))
      
      If inSelector
        
        ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_Selector, 0)
        ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, start, pos - start)
        
      Else
        
        If isKeyword(word, cssProperties)
          ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_Property, 0)
          ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, start, pos - start)
          
        ElseIf iskeyword(word, cssvalues)
          ScintillaSendMessage(gadget, #SCI_SETINDICATORCURRENT, #Style_CSS_Value, 0)
          ScintillaSendMessage(gadget, #SCI_INDICATORFILLRANGE, start, pos - start)
        EndIf
        
      EndIf
      
      Continue
    EndIf
    
    pos + 1
  Wend
EndProcedure

;-
;- Folding  

; Mettre à jour le folding (Pliage/Dépliage)
Procedure updateHTMLFolding(gadget, KeywordsFolding.s)  
  Protected lineCount, i    
  Protected level = 0
  Protected text.s 
  Protected buffer.s
  
  ; Traitement du pliage/depliage 
  ; Ligne par ligne depuis la premiére ligne
  
  ; Obtenir le nombre de lignes
  lineCount = ScintillaSendMessage(gadget, #SCI_GETLINECOUNT, 0, 0)
  
  ; Parcourir les lignes 
  For i = 0 To lineCount - 1
    
    ; Obtenir le texte de la ligne 
    text = GetScintillaLineText(gadget, i)
    
    ; Important : Supprimer #CRLF$
    text = Trim(RemoveString(text, #CRLF$))
    
    ; Supprimer les chaines entourant le déclencheur de folding
    ; Exemples "<html>" -> html ou "</html>" -> html 
    buffer = RemoveString(text, "<")
    buffer = RemoveString(buffer, ">")
    buffer = RemoveString(buffer, "/")
    
    ; Supprimer les espaces encadrant le texte
    buffer = Trim(buffer)
    
    ; Traitement balise de fermeture, on décremente aprés 
    If Bool(Left(text, 2) = "</" And isFoldingKeyWord(buffer, KeywordsFolding)) 
      ScintillaSendMessage(gadget, #SCI_SETFOLDLEVEL, i, #SC_FOLDLEVELBASE + level)
      level - 1
      If level < 0 : level = 0 : EndIf
      Continue
    EndIf
    
    ; Niveau normal
    ScintillaSendMessage(gadget, #SCI_SETFOLDLEVEL, i, #SC_FOLDLEVELBASE + level)
    
    ; Traitement balise ouvrante
    If Bool(Left(text, 1) = "<" And Mid(text, 2, 1) <> "/" And Right(text, 1)=">" And isFoldingKeyWord(buffer, KeywordsFolding))     
      ScintillaSendMessage(gadget, #SCI_SETFOLDLEVEL, i, #SC_FOLDLEVELBASE + level | #SC_FOLDLEVELHEADERFLAG)
      level + 1      
    EndIf
  Next
EndProcedure


;-
;- Utilitaire(s)

; Obtenir un UTF8 d'un string
Procedure makeUTF8Text(text.s)
  Static buffer.s
  
  buffer = Space(StringByteLength(text, #PB_UTF8) + 1)
  PokeS(@buffer, text, -1, #PB_UTF8)
  ProcedureReturn @buffer
EndProcedure

;-
;- Evenements
Procedure onTabBarClick()
  Protected *memoryID, size, textLength
  Protected tabItemPosition
  
  ; On quitte un onglet pour un autre onglet
  ; Chaque onglet posséde un attribut numérique (0 et 1)
  tabItemPosition = GetTabBarGadgetItemPosition(#tb, #TabBarGadgetItem_Selected)
  tabCurrentId = GetTabBarGadgetItemData(#tb, tabItemPosition)
  
  ; On quitte un panneau
  ;Select tabPreviousId
  ;  Case 0
  ;    currentProject\html = GetGadgetText(#editor)
  
  ;EndSelect
  
  tabPreviousId = tabCurrentId 
  
  Select EventType()      
    Case #TabBarGadget_EventType_Change            
      Select tabCurrentId
        Case 0 ; Edition de code HTML/CSS/JavaScript
          
          ; Suppression du filer du TabBarGadget
          SetTabBarGadgetAttribute(#tb, #TabBarGadget_BottomLine, #False)
          
          ; Activation de l'éditeur de code et désactivation du WebViewGadget
          HideGadget(#editor, #False)
          HideGadget(#webView, #True)
          
          ; Curseur de saisie dans l'éditeur de code
          SetActiveGadget(#editor)
          
        Case 1 ; Voir le résultat
          
          ; Un petit filet au bas du TabBarGadget (C'est plus joli)
          SetTabBarGadgetAttribute(#tb, #TabBarGadget_BottomLine, #True)
          
          ; Envoyer le code de l'éditeur vers le WebViewGadget
          SetGadgetItemText(#webView, #PB_WebView_HtmlCode, currentProject\html)
          
          ; Désactivation de l'éditeur de code et activation du WebViewGadget
          HideGadget(#editor, #True)
          HideGadget(#webView, #False)  
      EndSelect      
  EndSelect
EndProcedure

Procedure onResize()
  ;Resize tabbargadget
  ResizeGadget(#TB, #PB_Ignore, #PB_Ignore, WindowWidth(#app)-4, #PB_Ignore)
  UpdateTabBarGadget(#TB)  
  
  ResizeGadget(#editor, #PB_Ignore, #PB_Ignore, WindowWidth(#app), WindowHeight(#app)-MarginBot)
  ResizeGadget(#webView, #PB_Ignore, #PB_Ignore, WindowWidth(#app), WindowHeight(#app)-MarginBot)
EndProcedure

Procedure onClose()
  End
EndProcedure

; IDE Options = PureBasic 6.30 (Windows - x64)
; CursorPosition = 18
; Folding = 7---------------------
; EnableXP
; UseIcon = code.ico
; Executable = ..\Scintila pour les nuls\WebEditor1.exe
; CompileSourceDirectory
; EnablePurifier